## Explore asset allocations by age to determine rates of return in "other wealth" (excluding housing and super)

# Preliminaries -----------------------------------------------------------

rm(list=ls())
gc()



# Read HILDA data wave 18 ---------------------------------------------------------------

hilda_grouped <- qread("./Input data/Intermediate input data/hilda_grouped_master.qs") %>% 
  filter(wavenumber==18)


# Group assets ------------------------------------------------------------

## super
## housing

## other assets include:
## bank accounts
## cash investments - Government bonds, corporate bonds, debentures, certificates of deposit, and mortgage backed securities owned by the household.
## equity investments - Total shares, managed funds, and property trusts for the household.
## business investments
## trust funds
## life insurance policies
## collectibles and other assets - Total of substantial assets such as antiques, works of art, and collectibles for the household. Excludes home general contents
## vehicles

hilda_data_asset_alloc <- hilda_grouped %>% 
  ## other asset types
  mutate(cash_assets = pwtbani + pwcaini, ## bank and cash (cash is small)
         equity_assets = pweqini,
         ## "leftover assets" - unlikely to increase much in value (e.g. depreciation of business assets and vehicle) or small proportion of total assets
         leftover_assets = pwbusvi + pwtrusi + pwvechi +  pwinsui + pwcolli , # business, trust, vehicle, life insurance, collectibles (small)
         total_assets_check = super_assets + housing_assets + cash_assets + equity_assets + leftover_assets # check
         ) 

# check <- hilda_data_asset_alloc %>% filter( (total_assets_check- pwassei) > 1 | (total_assets_check- pwassei) < -1) ## total assets add up if 0 rows


## for each age group, get asset allocations excl super and housing
hilda_data_asset_alloc_age_summary <- hilda_data_asset_alloc %>% 
  group_by(age_grp) %>% 
  ## total assets held by age group
  summarise(across( contains("_assets") , ~sum(.x*hhwte) ),
            n = sum(hhwte)) %>% 
  ## share of "other assets"
  mutate(across(matches("_assets$"), ~ .x /other_assets , .names = "{.col}_share"))


## share plot excl super and housing
# asset_alloc_age_summary_share_plot <- ggplot(hilda_data_asset_alloc_age_summary %>% 
#                                                select(age_grp, matches("assets_share$"), 
#                                                       -matches("total_assets|housing_assets|super_assets|other_assets") ) %>%
#                                                pivot_longer(cols=matches("share$"), names_to="asset_type", values_to="asset_value" ) ) +
#   geom_col(aes(x=age_grp, y=asset_value, fill=asset_type))


## assume returns based off long term averages from Vanguard data 2000-2020
## https://insights.vanguard.com.au/static/asset-class/app.html
## cash 4.1%
## Australian shares 8.1%
## leftover assets approx half of cash rate 2.1% - assumed

## get the weighted average asset return figure
hilda_data_asset_alloc_age_return <- hilda_data_asset_alloc_age_summary %>% 
  mutate(other_assets_return_rate = cash_assets_share*0.041 + equity_assets_share*0.081 + leftover_assets_share*0.021) %>% 
  ## smooth it out via ma
  mutate(other_assets_return_rate_smooth = ma(other_assets_return_rate, 6) %>% as.numeric,
         ## for the young, replace NAs with unsmoothed rate - sample size large enough to be more reliable
         other_assets_return_rate_smooth = case_when(
           is.na(other_assets_return_rate_smooth) & age_grp<="[10,15)" ~ other_assets_return_rate,
           TRUE ~ other_assets_return_rate_smooth
         )) %>% 
  ## for the old, replace NAs with smoothed rates for 85-90 yos - sample size at older ages is small and asset allocations unlikely to change at old age
  fill(other_assets_return_rate_smooth, .direction="down") %>% 
  select(age_grp, other_assets_return_rate, other_assets_return_rate_smooth) 

# asset_alloc_age_return_plot <- ggplot(hilda_data_asset_alloc_age_return) +
#   geom_line(aes(x = as.numeric(age_grp), y = other_assets_return_rate_smooth))

## at young ages, assets are basically cash... as grow older, more of it is in depreciating business and vehicle assets...
## as get older, not much other assets but larger share in equities and cash

qsave(hilda_data_asset_alloc_age_return, "./Input data/other_asset_returns_a.qs")